<?php

/**
 * SwipeHQ Checkout extension for OpenCart 3.0.3.2+
 *
 * @version 3.0 / 18.02.202
 * @copyright (c) 2020, CyberCom.
 */
class ControllerExtensionPaymentSwipeHQ extends Controller {

    public function index() {

        $api_url = trim($this->config->get('payment_swipehq_api_url'), '/');
        $payment_page_url = trim($this->config->get('payment_swipehq_payment_page_url'), '/');

        $debug_mode = $this->config->get('payment_swipehq_debug_mode');

        $data['button_confirm'] = $this->language->get('button_confirm');
        $this->language->load('extension/payment/swipehq');

        $this->load->model('checkout/order');

        $order_info = $this->model_checkout_order->getOrder($this->session->data['order_id']);
        $currency = !empty($order_info['currency_code']) ? $order_info['currency_code'] : false;

        $acceptedCurrencies = $this->getAcceptedCurrencies();
        if (in_array($currency, $acceptedCurrencies)) {

            //get list of ordered products
            $products = '';
            $products_list = $this->cart->getProducts();
            foreach ($products_list as $product) {
                $products .= $product['quantity'] . ' x ' . $product['name'] . ($product != end($products_list) ? '<br />' : '');
            }

            $merchant = $this->config->get('payment_swipehq_merchant');
            $api_key = $this->config->get('payment_swipehq_api_key');

            $amount = $this->currency->convert($order_info['total'], $this->config->get('config_currency'), $currency);

            if (!empty($merchant) && !empty($api_key)) {
                //get product ID using TransactionIdentifier API
                $params = array(
                    'merchant_id' => $merchant,
                    'api_key' => $api_key,
                    'td_item' => 'Order ID' . $order_info['order_id'],
                    'td_description' => $products,
                    'td_amount' => $amount,
                    'td_currency' => $currency,
                    'td_default_quantity' => 1,
                    'td_user_data' => $order_info['order_id'],
                    'td_callback_url' => $this->url->link('extension/payment/swipehq/success'),
                    'td_lpn_url' => $this->url->link('extension/payment/swipehq/callback')
                );

                if (!empty($order_info['email'])) {
                    $params['td_email'] = $order_info['email'];
                }

                if ($debug_mode) {
                    $this->log->write('SwipeHQ Checkout debug: parameters for TransactionIdentifier API prepared: ' . json_encode($params));
                }

                $response = $this->_postToUrl($api_url . '/createTransactionIdentifier.php', $params);

                if ($response) {
                    if ($debug_mode) {
                        $this->log->write('SwipeHQ Checkout debug: response from TransactionIdentifier API: ' . $response);
                    }

                    $response_data = json_decode($response);
                    if ($response_data->response_code == 200 && !empty($response_data->data->identifier)) {
                        $data['action'] = $payment_page_url;
                        $data['identifier_id'] = $response_data->data->identifier;
                    } else {
                        $this->log->write('SwipeHQ Checkout error: wrong API call status, response code: ' . $response_data->response_code);
                        $data['error'] = 'Error: ' . $response_data->message;
                    }
                }
            } else {


                $this->log->write('SwipeHQ Checkout error: undefined Merchant ID or API Key');
            }
        } else {

            $this->log->write('SwipeHQ Checkout error: undefined currency code: ' . $currency);
            $data['error'] = '<strong style="color:red; font-size: 2em;">Error</strong><br/>Swipe does not support currency: ' . $currency . '. Swipe supports these currencies: ' . join(', ', $acceptedCurrencies) . '.';
        }

        return $this->load->view('extension/payment/swipehq', $data);
    }

    /**
     * Use  URL: /index.php?route=payment/payment_swipehq/success
     */
    public function success() {
        if ($this->config->get('payment_swipehq_debug_mode')) {
            $this->log->write('SwipeHQ Checkout debug: success page has been requested with parameters: ' . json_encode($_REQUEST));
        }

        $data = array();
        $request = $this->request->request;

        if (isset($request['result'])) {

            if ($request['result'] == 'accepted' || ($this->config->get('payment_swipehq_test_mode') == 1 && $request['result'] == 'test-accepted')) {

                $this->cart->clear();
                $this->response->redirect($this->url->link('checkout/success', '', 'SSL'));
                exit;
            } else {
                $this->log->write('SwipeHQ Checkout error: transaction has been declined.');
                $this->response->redirect($this->url->link('checkout/failure', '', 'SSL'));
            }
        } else {
            $this->log->write('SwipeHQ Checkout error: success page has been requested with undefined result of payment.');
        }
        $this->redirect($this->url->link('checkout/checkout'));
    }

    /**
     * Use Callback URL: /index.php?route=payment/payment_swipehq/callback
     */
    public function callback() {
        if ($this->config->get('payment_swipehq_debug_mode')) {
            $this->log->write('SwipeHQ Checkout debug: callback page has been requested with parameters: ' . json_encode($_REQUEST));
        }

        $request = $this->request->request;

        if (!empty($request['td_user_data']) && is_numeric($request['td_user_data'])) {
            $order_id = $request['td_user_data'];
            $this->load->model('checkout/order');

            if ($request['status'] == 'test-accepted' && $this->config->get('payment_swipehq_test_mode') == 1) {
                $status = $this->config->get('payment_swipehq_test_order_status_id');
                $this->model_checkout_order->addOrderHistory($order_id, $status, $request['status'], false);
                /*
                if (version_compare(VERSION, '2.0.0.0') >= 0) {
                    $this->model_checkout_order->addOrderHistory($order_id, $status, $request['status'], false);
                } else {
                    $this->model_checkout_order->confirm($order_id, $status);
                } */

                if ($this->config->get('payment_swipehq_debug_mode')) {
                    $this->log->write('SwipeHQ Checkout debug: test mode enabled, status of order ID' . $order_id . ' has been changed to ' . $status);
                }
            } else {
                $is_verified = $this->_verifyTransaction($request['transaction_id']);
                if ($is_verified) {
                    
                    if ($request['status'] == 'accepted') {
                        $status = $this->config->get('payment_swipehq_order_status_id');
                    } else {
                        $this->log->write('SwipeHQ Checkout error: payment has not been accepted. Order ID: ' . $order_id);
                        $status = $this->config->get('payment_swipehq_failed_order_status_id');
                    }
                    
                    $this->model_checkout_order->addOrderHistory($order_id, $status, $request['status'], false);
                    
                    /*
                    if (version_compare(VERSION, '2.0.0.0') >= 0) {
                        $this->model_checkout_order->addOrderHistory($order_id, $status, $request['status'], false);
                    } else {
                        $this->model_checkout_order->confirm($order_id, $status);
                    }*/
                    
                    if ($this->config->get('payment_swipehq_debug_mode')) {
                        $this->log->write('SwipeHQ Checkout debug: status of order ID' . $order_id . ' has been changed to ' . $status);
                    }
                } else {
                    $this->log->write('SwipeHQ Checkout error: verification of transaction ID ' . $request['transaction_id'] . ' has been failed, order ID: ' . $order_id);
                }
            }
        } else {
            $this->log->write('SwipeHQ Checkout error: undefined order ID in the LPN request (empty "td_user_data").');
        }
    }

    /**
     * Transaction verification
     *
     * @param string $transaction_id
     * @return boolean
     */
    private function _verifyTransaction($transaction_id) {

        $api_url = trim($this->config->get('payment_swipehq_api_url'), '/');

        //parameters for transaction verification
        $data = array(
            'merchant_id' => $this->config->get('payment_swipehq_merchant'),
            'api_key' => $this->config->get('payment_swipehq_api_key'),
            'transaction_id' => $transaction_id
        );

        if ($this->config->get('payment_swipehq_debug_mode')) {
            $this->log->write('SwipeHQ Checkout debug: parameters for transaction verification prepared: ' . json_encode($data));
        }

        $response = $this->_postToUrl($api_url . '/verifyTransaction.php', $data);
        if (!empty($response)) {
            if ($this->config->get('payment_swipehq_debug_mode')) {
                $this->log->write('SwipeHQ Checkout debug: response from VerifyTransaction API: ' . $response);
            }

            $response_data = json_decode($response);

            if ($response_data->response_code == 200 && $response_data->data->transaction_id == $transaction_id) {
                return true;
            }
        }
        return false;
    }

    private function _postToUrl($url, $data) {
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
        $response = curl_exec($ch);
        curl_close($ch);

        if (!$response) {
            $this->log->write('SwipeHQ Checkout error: CURL request error: ' . curl_error($ch) . '(' . curl_errno($ch) . ')');
        }

        return $response;
    }

    protected function getAcceptedCurrencies() {
        $api_url = trim($this->config->get('payment_swipehq_api_url'), '/');

        $url = $api_url . '/fetchCurrencyCodes.php';
        $data = array(
            'merchant_id' => $this->config->get('payment_swipehq_merchant'),
            'api_key' => $this->config->get('payment_swipehq_api_key'),
        );

        if ($this->config->get('payment_swipehq_debug_mode')) {
            $this->log->write('Swipe - fetchCurrencyCodes request: ' . $url . ' - ' . print_r($data, true));
        }

        $response = $this->_postToUrl($url, $data);

        if ($this->config->get('payment_swipehq_debug_mode')) {
            $this->log->write('Swipe - fetchCurrencyCodes response: ' . print_r($response, true));
        }

        $response_arr = json_decode($response, true);
        return $response_arr['data'];
    }

}

